<?php
/**
 * Elementor Widget
 * @package Bizmaster
 * @since 1.0.0
 */

namespace Elementor;
class Bizmaster_Feature_Widget extends Widget_Base
{
    /**
     * Get widget name.
     *
     * Retrieve Elementor widget name.
     *
     * @return string Widget name.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_name()
    {
        return 'bizmaster-feature-widget';
    }

    /**
     * Get widget title.
     *
     * Retrieve Elementor widget title.
     *
     * @return string Widget title.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_title()
    {
        return esc_html__('Feature', 'bizmaster-core');
    }

    public function get_keywords()
    {
        return ['Section', 'Heading', 'Title', "HugeBinary", 'Bizmaster'];
    }

    /**
     * Get widget icon.
     *
     * Retrieve Elementor widget icon.
     *
     * @return string Widget icon.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_icon()
    {
        return 'eicon-post-list';
    }

    /**
     * Get widget categories.
     *
     * Retrieve the list of categories the Elementor widget belongs to.
     *
     * @return array Widget categories.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_categories()
    {
        return ['bizmaster_widgets'];
    }

    /**
     * Register Elementor widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function register_controls()
	{
		$this->start_controls_section(
			'settings_section',
			[
				'label' => esc_html__('General Settings', 'bizmaster-core'),
				'tab' => Controls_Manager::TAB_CONTENT
			]
		);

        $repeater = new \Elementor\Repeater();
		$repeater->add_control(
			'title',
			[
				'label' => esc_html__('Title', 'bizmaster-core'),
				'type' => Controls_Manager::TEXT,
				'default' => esc_html__('Business Growth Catalyst', 'bizmaster-core')
			]
		);

        $repeater->add_control(
			'content',
			[
				'label' => esc_html__('Content', 'bizmaster-core'),
				'type' => Controls_Manager::TEXTAREA,
				'default' => esc_html__('Various versions have evolved over the years, sometimes by accident, sometimes on purpose.', 'bizmaster-core')
			]
		);

        $repeater->add_control(
            'btn_text',
			[
                'label' => esc_html__('Button Text', 'bizmaster-core'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Read More', 'bizmaster-core')
			]
		);

		$repeater->add_control(
			'btn_url',
			[
				'label' => esc_html__( 'Button Url', 'bizmaster-core' ),
				'type' => Controls_Manager::URL,
				'options' => [ 'url', 'is_external', 'nofollow' ],
				'default' => [
					'url' => '#',
					'is_external' => false,
					'nofollow' => false
				],
				'label_block' => true
			]
		);

        $this->add_control('feature_items', [
			'label' => esc_html__('Feature Items', 'bizmaster-core'),
			'type' => Controls_Manager::REPEATER,
			'fields' => $repeater->get_controls()
		]);

		$this->end_controls_section();

        /*  tab styling tabs start */
		$this->start_controls_section(
			'style_settings_section',
			[
				'label' => esc_html__('Style Settings', 'bizmaster-core'),
				'tab' => Controls_Manager::TAB_STYLE
			]
		);

		$this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name'     => 'border_color',
				'label'    => esc_html__('Card Border', 'bizmaster-core'),
				'selector' => '{{WRAPPER}} .feature-card'
			]
		);

		$this->add_control(
			'card_border_radius',
			[
				'label'      => esc_html__('Card Border Radius', 'bizmaster-core'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .feature-card' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

        $this->add_control('bg_color', [
			'label' => esc_html__('Card BG Color', 'bizmaster-core'),
			'type' => Controls_Manager::COLOR,
			'selectors' => [
				"{{WRAPPER}} .feature-card" => "background-color: {{VALUE}}"
			]
		]);

		$this->add_control('bg_hover_color', [
			'label' => esc_html__('Card BG Hover Color', 'bizmaster-core'),
			'type' => Controls_Manager::COLOR,
			'selectors' => [
				"{{WRAPPER}} .feature-card:hover" => "background-color: {{VALUE}}",
				"{{WRAPPER}} .feature-card:hover" => "border-color: {{VALUE}}"
			]
		]);

		$this->add_responsive_control(
			'card_padding',
			[
				'label'      => esc_html__('Card Padding', 'bizmaster-core'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .feature-card' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]
		);

		$this->add_control('title_color', [
			'label' => esc_html__('Title Color', 'bizmaster-core'),
			'type' => Controls_Manager::COLOR,
			'selectors' => [
				"{{WRAPPER}} .feature-card .feature-card-title" => "color: {{VALUE}}"
			]
		]);

		$this->add_group_control(Group_Control_Typography::get_type(), [
			'label' => esc_html__('Title Typography', 'bizmaster-core'),
			'name' => 'title_typography',
			'description' => esc_html__('Title Typography', 'bizmaster-core'),
			'selector' => "{{WRAPPER}} .feature-card .feature-card-title"
		]);

		$this->add_control('content_color', [
			'label' => esc_html__('Content Color', 'bizmaster-core'),
			'type' => Controls_Manager::COLOR,
			'selectors' => [
				"{{WRAPPER}} .feature-card .feature-card-text" => "color: {{VALUE}}"
			]
		]);

		$this->add_group_control(Group_Control_Typography::get_type(), [
			'label' => esc_html__('Content Typography', 'bizmaster-core'),
			'name' => 'content_typography',
			'description' => esc_html__('Content Typography', 'bizmaster-core'),
			'selector' => "{{WRAPPER}} .feature-card .feature-card-text"
		]);

		$this->add_control('btn_color', [
            'label' => esc_html__('Button Color', 'bizmaster-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .feature-card .global-btn.style-border" => "color: {{VALUE}}"
            ]
        ]);

		$this->add_control('btn_hover_color', [
            'label' => esc_html__('Button Hover Color', 'bizmaster-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
				"{{WRAPPER}} .feature-card .global-btn.style-border:hover" => "color: {{VALUE}}"
			]
		]);

		$this->add_control('btn_border_color', [
            'label' => esc_html__('Button Border Color', 'bizmaster-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
				"{{WRAPPER}} .feature-card .global-btn.style-border" => "border-color: {{VALUE}}"
			]
		]);

		$this->add_control('btn_border_hover_color', [
            'label' => esc_html__('Button Border Hover Color', 'bizmaster-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
				"{{WRAPPER}} .feature-card .global-btn.style-border:hover" => "border-color: {{VALUE}}"
			]
		]);

		$this->add_control('btn_bg_color', [
            'label' => esc_html__('Button BG Color', 'bizmaster-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .feature-card .global-btn.style-border" => "background-color: {{VALUE}}"
			]
		]);

		$this->add_control('btn_hover_bg_color', [
            'label' => esc_html__('Button Hover BG Color', 'bizmaster-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .feature-card .global-btn.style-border:hover:before" => "background-color: {{VALUE}}",
                "{{WRAPPER}} .feature-card .global-btn.style-border:hover:after" => "background-color: {{VALUE}}"
			]
		]);

		$this->end_controls_section();
	}

    /**
     * Render Elementor widget output on the frontend.
     *
     * Written in PHP and used to generate the final HTML.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function render()
	{
		$settings = $this->get_settings_for_display();
		$feature_items = $settings['feature_items'];
		?>
		<div class="feature-area-4">
			<div class="row gy-40 justify-content-center">
				<?php foreach ($feature_items as $index => $item): ?>
					<div class="col-lg-12 col-md-6">
						<div class="feature-card">
							<h4 class="feature-card-title"><?php echo $item['title']; ?></h4>
							<p class="feature-card-text"><?php echo $item['content']; ?></p>
							<?php if(!empty($item['btn_text'])) : ?>
								<?php
									if ( ! empty( $item['btn_url']['url'] ) ) {
										$this->add_link_attributes( "btn_url{$index}", $item['btn_url'] );
									}
								?>
								<div class="btn-wrap">
									<a class="global-btn style-border" <?php $this->print_render_attribute_string( "btn_url{$index}" ); ?>>
										<?php echo $item['btn_text']; ?> <i class="fas fa-arrow-right ms-2"></i>
									</a>
								</div>
							<?php endif; ?>
						</div>
					</div>
				<?php endforeach; ?>
			</div>
		</div>
		<?php
	}
}

Plugin::instance()->widgets_manager->register_widget_type(new Bizmaster_Feature_Widget());
